import * as common from '/pages/src/common.mjs';

let userOpacity = 90;
let userZoom = 100;
let currentTab = 'team'; 

// --- GESTION ONGLETS ---
window.switchTab = function(tab) {
    currentTab = tab;
    
    document.getElementById('tab-team').className = tab === 'team' ? 'tab-btn active' : 'tab-btn';
    document.getElementById('tab-indiv').className = tab === 'indiv' ? 'tab-btn active' : 'tab-btn';
    
    if (tab === 'team') {
        document.getElementById('list-team').classList.remove('hidden');
        document.getElementById('list-indiv').classList.add('hidden');
    } else {
        document.getElementById('list-team').classList.add('hidden');
        document.getElementById('list-indiv').classList.remove('hidden');
    }
}

// --- REGLAGES ---
function loadSettings() {
    const sOp = localStorage.getItem('myMod_Team_Op');
    const sZoom = localStorage.getItem('myMod_Team_Zoom');

    if(sOp) userOpacity = parseInt(sOp);
    if(sZoom) userZoom = parseInt(sZoom);

    document.getElementById('in-opacity').value = userOpacity;
    document.getElementById('in-zoom').value = userZoom;
    applyVisuals();
}

function saveSettings() {
    userOpacity = parseInt(document.getElementById('in-opacity').value);
    userZoom = parseInt(document.getElementById('in-zoom').value);
    
    localStorage.setItem('myMod_Team_Op', userOpacity);
    localStorage.setItem('myMod_Team_Zoom', userZoom);

    applyVisuals();
    document.getElementById('settings-panel').style.display = 'none';
}

function applyVisuals() {
    document.getElementById('op-label').innerText = userOpacity;
    document.getElementById('dash').style.backgroundColor = `rgba(0, 0, 0, ${userOpacity/100})`;
    document.getElementById('zoom-label').innerText = (userZoom/100).toFixed(1);
    document.getElementById('wrapper').style.transform = `scale(${userZoom/100})`;
}

// --- UTILITAIRES ---
function formatGap(sec) {
    if (Math.abs(sec) < 0.5) return '';
    const sign = sec > 0 ? '+' : '-';
    const color = sec > 0 ? '#ff5555' : '#55ff55';
    return `<span style="color:${color}">${sign}${Math.round(Math.abs(sec))}s</span>`;
}

function formatName(first, last) {
    if (!last) return first || 'Inconnu';
    const fInitial = first ? first.charAt(0).toUpperCase() + '.' : '';
    return `${fInitial} ${last.toUpperCase()}`;
}

// Nouvelle fonction pour nettoyer et regrouper les noms d'équipes
function normalizeTeamName(name) {
    if (!name || name.trim() === "") return "Indépendant";
    
    // 1. Tout passer en majuscules pour éviter les doublons liés à la casse
    let norm = name.toUpperCase().trim();
    
    // 2. Supprimer les tags de catégorie à la fin, ex: "TEAM (C)", "TEAM [A]"
    norm = norm.replace(/\s*[([][A-E][)\]]\s*$/, '');
    
    return norm.trim() || "Indépendant";
}

// --- LOGIQUE METIER ---
function updateRanking(data) {
    let riders = [];
    if (Array.isArray(data)) riders = data;
    else if (data && data.riders) riders = data.riders;
    else if (data && data.nearby) riders = data.nearby;

    if (!riders || riders.length === 0) return;

    // TRI PAR GAP (Position route)
    riders.sort((a, b) => (a.gap || 0) - (b.gap || 0));

    let teams = {};
    let indivHtml = '';

    riders.forEach((r, index) => {
        // --- DONNEES ---
        // On utilise la nouvelle fonction pour obtenir un nom propre et unique
        let rawTeam = (r.athlete && r.athlete.team) ? r.athlete.team : "";
        let teamName = normalizeTeamName(rawTeam);
        
        let name = "Inconnu";
        if (r.athlete) name = formatName(r.athlete.firstName, r.athlete.lastName);

        const position = index + 1;
        const isMe = r.watching;

        // --- CALCUL EQUIPE ---
        if (!teams[teamName]) {
            teams[teamName] = { name: teamName, score: 0, count: 0, isMyTeam: false };
        }
        teams[teamName].score += position;
        teams[teamName].count += 1;
        if (isMe) teams[teamName].isMyTeam = true;

        // --- HTML INDIVIDUEL ---
        let rowClass = isMe ? 'row indiv-row watching' : 'row indiv-row';
        indivHtml += `
            <div class="${rowClass}">
                <div class="i-pos">${position}</div>
                <div class="i-name" title="${name}">${name}</div>
                <div class="i-team" title="${teamName}">${teamName.substring(0,12)}</div>
                <div class="i-gap">${formatGap(r.gap || 0)}</div>
            </div>
        `;
    });

    // --- HTML EQUIPE ---
    let ranking = Object.values(teams);
    ranking.sort((a, b) => a.score - b.score);

    let teamHtml = '';
    ranking.forEach((t, i) => {
        let style = t.isMyTeam ? 'color: #fc6719;' : '';
        teamHtml += `
            <div class="row team-row" style="${style}">
                <div class="t-rank">${i + 1}</div>
                <div class="t-name" title="${t.name}">
                    ${t.name} <span class="riders-count">(${t.count})</span>
                </div>
                <div class="t-score">${t.score}</div>
            </div>
        `;
    });

    document.getElementById('list-team').innerHTML = teamHtml || '<div class="msg">Aucune donnée</div>';
    document.getElementById('list-indiv').innerHTML = indivHtml || '<div class="msg">Aucune donnée</div>';
}

// --- MAIN ---
export async function main() {
    loadSettings();

    document.getElementById('btn-settings').addEventListener('click', () => {
        const p = document.getElementById('settings-panel');
        p.style.display = p.style.display === 'block' ? 'none' : 'block';
    });
    document.getElementById('btn-save').addEventListener('click', saveSettings);
    
    document.getElementById('in-opacity').addEventListener('input', (e) => {
        userOpacity = e.target.value; applyVisuals();
    });
    document.getElementById('in-zoom').addEventListener('input', (e) => {
        userZoom = e.target.value; applyVisuals();
    });

    if (common && common.initInteractionListeners) {
        common.initInteractionListeners();
        common.subscribe('nearby', updateRanking);
    }
}