import * as common from '/pages/src/common.mjs';
import * as o101Common from './o101/common.mjs'; 

let userOpacity = 90;
let userZoom = 100;

// Cache pour optimiser (comme dans progress-bar de o101)
const courseInfo = {
    event: {
        activeEvent: null,
        eventSubgroupId: 0
    },
    totalDistance: 0,
    progress: 0,
    distanceCompleted: 0,
    lapInfo: {
        laps: 0,
        lapDistance: 0
    },
    leadInDistance: 0,
    distanceToFinish: 0
};

// --- REGLAGES ---
function loadSettings() {
    const sOp = localStorage.getItem('myMod_Infos_Op');
    const sZoom = localStorage.getItem('myMod_Infos_Zoom');

    if(sOp) userOpacity = parseInt(sOp);
    if(sZoom) userZoom = parseInt(sZoom);

    document.getElementById('in-opacity').value = userOpacity;
    document.getElementById('in-zoom').value = userZoom;
    
    applyVisuals();
}

function saveSettings() {
    const o = document.getElementById('in-opacity').value;
    const z = document.getElementById('in-zoom').value;

    userOpacity = parseInt(o); localStorage.setItem('myMod_Infos_Op', userOpacity);
    userZoom = parseInt(z); localStorage.setItem('myMod_Infos_Zoom', userZoom);

    applyVisuals();
    document.getElementById('settings-panel').style.display = 'none';
}

function applyVisuals() {
    document.getElementById('op-label').innerText = userOpacity;
    document.getElementById('dash').style.backgroundColor = `rgba(0, 0, 0, ${userOpacity/100})`;
    
    document.getElementById('zoom-label').innerText = (userZoom/100).toFixed(1);
    document.getElementById('wrapper').style.transform = `scale(${userZoom/100})`;
}

// --- UTILITAIRES ---
function formatTime(seconds) {
    if (!seconds) return '00:00';
    const h = Math.floor(seconds / 3600);
    const m = Math.floor((seconds % 3600) / 60);
    const s = Math.floor(seconds % 60);
    
    const mStr = m.toString().padStart(2, '0');
    const sStr = s.toString().padStart(2, '0');
    
    if (h > 0) return `${h}:${mStr}:${sStr}`;
    return `${mStr}:${sStr}`;
}

// Helper pour récupérer l'ID de sous-groupe (comme o101)
const getEventSubgroupId = (athlete) => {
    let eventSubgroupId;
    // Si o101 est présent dans l'objet athlete (dépend de la version de Sauce)
    if (Object.hasOwn(athlete, 'o101')) {
        eventSubgroupId = athlete.o101.eventSubgroupId;
    }
    return eventSubgroupId ?? athlete.state.eventSubgroupId;
}


// --- MISE A JOUR ---
async function updateDisplay(info) {
    if (!info || !info.state) return;
    const state = info.state;

    // 1. Données Physiques
    const speedKmh = (state.speed || 0).toFixed(1); 
    const distKm = ((state.distance || 0) / 1000).toFixed(1);
    const elev = Math.round(state.climbing || 0); 
    const time = formatTime(state.time || 0);

    document.getElementById('val-speed').innerText = speedKmh;
    document.getElementById('val-dist').innerText = distKm;
    document.getElementById('val-elev').innerText = elev;
    document.getElementById('val-time').innerText = time;


    // 2. Progression (Logique importée de progress-bar.mjs)
    
    const event = courseInfo.event;
    // On passe l'objet complet 'info' (packet) à getEventSubgroupId
    const eventSubgroupId = getEventSubgroupId(info);
   
    // Mise à jour des infos d'event si ça change
    if (event.activeEvent == null || eventSubgroupId != event.eventSubgroupId) {
        event.eventSubgroupId = eventSubgroupId;
        event.activeEvent = await o101Common.lazyGetSubgroup(eventSubgroupId);
    }

    if (event.activeEvent == null) {
        // --- MODE ROUTE LIBRE ---
        let route = await o101Common.lazyGetRoute(state.routeId);

        if (route) {
            const routeDistance = route.distanceInMeters;
            courseInfo.totalDistance = routeDistance;
            courseInfo.leadInDistance = route.leadinDistanceInMeters || 0;
            
            let progDist = state.eventDistance || (state.distance % routeDistance);
            
            courseInfo.distanceCompleted = progDist;
            courseInfo.distanceToFinish = Math.max(0, routeDistance - progDist);
            
            if (routeDistance > 0) {
                courseInfo.progress = (progDist / routeDistance) * 100;
            } else {
                courseInfo.progress = 0;
            }
        }
    } else {
        // --- MODE EVENEMENT ---
        let zwiftRoute = await o101Common.lazyGetRoute(event.activeEvent.routeId);

        if (zwiftRoute) {
            const routeDistance = event.activeEvent.distanceInMeters ?? event.activeEvent.routeDistance;

            courseInfo.totalDistance = routeDistance;
            courseInfo.lapInfo.laps = event.activeEvent.laps;
            courseInfo.leadInDistance = zwiftRoute.leadinDistanceInMeters || 0;
            
            courseInfo.distanceCompleted = state.eventDistance;
            courseInfo.distanceToFinish = Math.max(0, routeDistance - state.eventDistance);
            
            if (routeDistance > 0) {
                courseInfo.progress = (state.eventDistance / routeDistance) * 100;
            } else {
                courseInfo.progress = 0;
            }
        }
    }

    // --- AFFICHAGE BARRE ---
    const progBar = document.getElementById('prog-bar-fill');
    const progText = document.getElementById('prog-text');

    if (courseInfo.totalDistance > 0) {
        // Bornage 0-100
        let pct = Math.min(100, Math.max(0, courseInfo.progress));
        let remainKm = (courseInfo.distanceToFinish / 1000).toFixed(1);

        progBar.style.width = `${pct}%`;
        
        let label = (event.activeEvent) ? 'Arrivée :' : 'Fin tour :';
        progText.innerHTML = `${label} <b>${remainKm} km</b> (${pct.toFixed(1)}%)`;
        
        // Couleur
        if (pct > 95) progBar.style.backgroundColor = '#fc6719';
        else progBar.style.backgroundColor = '#fc6719';

    } else {
        progBar.style.width = '0%';
        progText.innerHTML = '<span style="opacity:0.5">Course libre / Pas de cible</span>';
        progBar.style.backgroundColor = '#555';
    }
}

// --- MAIN ---
export async function main() {
    loadSettings();

    document.getElementById('btn-settings').addEventListener('click', () => {
        const p = document.getElementById('settings-panel');
        p.style.display = p.style.display === 'block' ? 'none' : 'block';
    });
    document.getElementById('btn-save').addEventListener('click', saveSettings);
    
    document.getElementById('in-opacity').addEventListener('input', (e) => {
        userOpacity = e.target.value; applyVisuals();
    });
    document.getElementById('in-zoom').addEventListener('input', (e) => {
        userZoom = e.target.value; applyVisuals();
    });

    if (common && common.initInteractionListeners) {
        common.initInteractionListeners();
        
        // On s'abonne. Le packet entier est passé à updateDisplay pour avoir accès à tout.
        common.subscribe('athlete/watching', (packet) => {
            if(packet) {
                updateDisplay(packet); 
            }
        });
    }
}