import * as common from '/pages/src/common.mjs';
import * as o101 from './o101/common.mjs';

let settings = {
    opacity: 90,
    zoom: 100,
    groupGapThreshold: 4,
    refreshRate: 1,
    nearbyCount: 4 // Par défaut : 4 groupes autour de moi
};

let lastUpdate = 0; 

// --- REGLAGES ---
function loadSettings() {
    const sOp = localStorage.getItem('myMod_DS_Op');
    const sZoom = localStorage.getItem('myMod_DS_Zoom');
    const sRef = localStorage.getItem('myMod_DS_Refresh'); 
    const sNear = localStorage.getItem('myMod_DS_Nearby'); 
    
    if(sOp) settings.opacity = parseInt(sOp);
    if(sZoom) settings.zoom = parseInt(sZoom);
    if(sRef) settings.refreshRate = parseInt(sRef);
    if(sNear) settings.nearbyCount = parseInt(sNear);

    document.getElementById('in-opacity').value = settings.opacity;
    document.getElementById('in-zoom').value = settings.zoom;
    document.getElementById('in-refresh').value = settings.refreshRate;
    document.getElementById('in-nearby').value = settings.nearbyCount;

    document.getElementById('refresh-label').innerText = settings.refreshRate + 's';
    document.getElementById('nearby-label').innerText = settings.nearbyCount;
    
    applyVisuals();
}

function saveSettings() {
    settings.opacity = parseInt(document.getElementById('in-opacity').value);
    settings.zoom = parseInt(document.getElementById('in-zoom').value);
    settings.refreshRate = parseInt(document.getElementById('in-refresh').value);
    settings.nearbyCount = parseInt(document.getElementById('in-nearby').value);
    
    localStorage.setItem('myMod_DS_Op', settings.opacity);
    localStorage.setItem('myMod_DS_Zoom', settings.zoom);
    localStorage.setItem('myMod_DS_Refresh', settings.refreshRate);
    localStorage.setItem('myMod_DS_Nearby', settings.nearbyCount);

    applyVisuals();
    document.getElementById('settings-panel').style.display = 'none';
}

function applyVisuals() {
    document.getElementById('op-label').innerText = settings.opacity;
    document.getElementById('dash').style.backgroundColor = `rgba(0, 0, 0, ${settings.opacity/100})`;
    document.getElementById('zoom-label').innerText = (settings.zoom/100).toFixed(1);
    document.getElementById('wrapper').style.transform = `scale(${settings.zoom/100})`;
}

// --- LOGIQUE METIER ---

// Fonction pour nettoyer et regrouper les noms d'équipes (Casse + Catégories)
function normalizeTeamName(name) {
    if (!name || name.trim() === "") return "";
    
    // 1. Tout passer en majuscules pour éviter les doublons
    let norm = name.toUpperCase().trim();
    
    // 2. Supprimer les tags de catégorie à la fin, ex: "TEAM (C)", "TEAM [A]"
    norm = norm.replace(/\s*[([][A-E][)\]]\s*$/, '');
    
    return norm.trim();
}

function getTeamColor(teamName) {
    if (!teamName) return '#ccc';
    if (o101 && o101.preferredTeamColor) {
        const tc = o101.preferredTeamColor(teamName);
        if (tc && tc.color && tc.color !== '#FFF') return tc.color;
    }
    return stringToColorHSL(teamName);
}

function stringToColorHSL(str) {
    let hash = 0;
    for (let i = 0; i < str.length; i++) hash = str.charCodeAt(i) + ((hash << 5) - hash);
    return `hsl(${Math.abs(hash) % 360}, 75%, 65%)`;
}

function formatTime(sec) {
    if (sec == null || isNaN(sec)) return '';
    const abs = Math.abs(sec);
    if (abs < 60) return Math.round(abs) + 's';
    const m = Math.floor(abs / 60);
    const s = Math.floor(abs % 60);
    return `${m}:${s.toString().padStart(2,'0')}`;
}

function updateGroups(data) {
    const now = Date.now();
    if (now - lastUpdate < settings.refreshRate * 1000) return;
    lastUpdate = now;

    let riders = [];
    if (Array.isArray(data)) riders = data;
    else if (data && data.nearby) riders = data.nearby;
    else return;

    if (riders.length === 0) {
        document.getElementById('groups-container').innerHTML = '<div class="msg">Aucune donnée...</div>';
        return;
    }

    // 1. Tri des coureurs par position (gap)
    riders.sort((a, b) => (a.gap || 0) - (b.gap || 0));

    // 2. Création des groupes
    let groups = [];
    let myGroupIndex = -1; 

    if (riders.length > 0) {
        let currentGroup = [riders[0]];
        for (let i = 1; i < riders.length; i++) {
            const r = riders[i];
            const prev = riders[i-1];
            const diff = (r.gap || 0) - (prev.gap || 0);

            if (diff > settings.groupGapThreshold) {
                if (currentGroup.some(r => r.watching)) myGroupIndex = groups.length;
                groups.push(currentGroup);
                currentGroup = [r];
            } else {
                currentGroup.push(r);
            }
        }
        if (currentGroup.some(r => r.watching)) myGroupIndex = groups.length;
        groups.push(currentGroup);
    }

    // Calcul des moyennes de gap pour stabilisation
    const groupsAvgGap = groups.map(grp => {
        const sum = grp.reduce((acc, r) => acc + (r.gap || 0), 0);
        return sum / grp.length;
    });

    let myGroupGapRef = 0;
    if (myGroupIndex !== -1) {
        myGroupGapRef = groupsAvgGap[myGroupIndex];
    } else {
        // Si je ne suis pas trouvé, on centre sur le milieu
        myGroupIndex = Math.floor(groups.length / 2);
    }

    // --- SELECTION DES GROUPES A AFFICHER ---
    let indicesToShow = new Set();
    indicesToShow.add(0); // Tête de course
    indicesToShow.add(groups.length - 1); // Queue de course

    // Utilisation du paramètre dynamique 'nearbyCount'
    const scope = settings.nearbyCount;
    for (let i = myGroupIndex - scope; i <= myGroupIndex + scope; i++) {
        if (i >= 0 && i < groups.length) {
            indicesToShow.add(i);
        }
    }

    // Conversion en tableau trié
    let sortedIndices = Array.from(indicesToShow).sort((a, b) => a - b);

    // 3. Construction HTML avec "trous"
    let html = '';
    
    for (let k = 0; k < sortedIndices.length; k++) {
        const index = sortedIndices[k];
        const group = groups[index];

        // Vérifier s'il y a un trou avec l'index précédent affiché
        if (k > 0) {
            const prevIndexShown = sortedIndices[k - 1];
            const gapIndices = index - prevIndexShown - 1; // Nombre de groupes sautés
            
            if (gapIndices > 0) {
                // Message des groupes intermédiaires
                html += `
                    <div style="text-align:center; padding:5px; background:rgba(255,255,255,0.05); margin:2px 0; font-size:11px; color:#888; border-radius:4px;">
                        ... ${gapIndices} groupe(s) intermédiaire(s) ...
                    </div>
                    <div class="link-arrow">▼</div>
                `;
            } else {
                html += `<div class="link-arrow">▼</div>`;
            }
        }

        // --- RENDU DE LA CARTE GROUPE ---
        let totalWkg = 0;
        let teamsMap = new Map();
        let watchingInGroup = false;

        group.forEach(r => {
            const pwr = (r.state && r.state.power) || 0;
            const wgt = (r.athlete && r.athlete.weight) || 75;
            totalWkg += (pwr / wgt);

            // GESTION ET NORMALISATION DES ÉQUIPES
            if (r.athlete && r.athlete.team) {
                let t = r.athlete.team;
                if (Array.isArray(t)) t = t[0];
                
                const tName = normalizeTeamName(t); // Appel de notre fonction de nettoyage
                if (tName) {
                    teamsMap.set(tName, (teamsMap.get(tName) || 0) + 1);
                }
            }
            if (r.watching) watchingInGroup = true;
        });

        const avgWkg = (totalWkg / group.length).toFixed(1);
        const sortedTeams = [...teamsMap.entries()].sort((a, b) => b[1] - a[1]);

        let gapText = '';
        let gapClass = '';
        let myRelGapHtml = ''; 

        if (index === 0) {
            gapText = 'TÊTE';
            gapClass = 'gap-head';
        } else {
            const diff = groupsAvgGap[index] - groupsAvgGap[index - 1];
            gapText = '+ ' + formatTime(diff);
            gapClass = 'gap-chase';
        }

        if (!watchingInGroup && myGroupIndex !== -1) {
            const diffWithMe = groupsAvgGap[index] - myGroupGapRef;
            const absDiff = Math.abs(diffWithMe);
            const sign = diffWithMe > 0 ? '+' : '-'; 
            const relText = sign + ' ' + formatTime(absDiff);
            const relColor = diffWithMe > 0 ? '#ff5555' : '#44ff44'; 
            
            myRelGapHtml = `<div class="my-gap-badge" style="color:${relColor}">VS MOI : ${relText}</div>`;
        } else if (watchingInGroup) {
             myRelGapHtml = `<div class="my-gap-badge" style="color:#fc6719; font-weight:bold;">MON GROUPE</div>`;
        }

        let cardClass = 'group-card';
        if (watchingInGroup) cardClass += ' watching-group';

        html += `
        <div class="${cardClass}">
            <div class="group-header">
                <div class="group-info">
                    <span class="group-count">${group.length} 👤</span>
                    <span class="group-wkg">${avgWkg} w/kg</span>
                </div>
                <div class="group-gaps">
                    <div class="${gapClass}">${gapText}</div>
                    ${myRelGapHtml}
                </div>
            </div>
            
            <div class="team-list">
        `;

        if (sortedTeams.length === 0) {
             html += `<div style="font-style:italic; color:#666; font-size:10px;">Individuels</div>`;
        } else {
            sortedTeams.slice(0, 10).forEach(([name, count]) => {
                const color = getTeamColor(name);
                const bg = color + '22'; 
                // 3 premières lettres
                const shortName = name.substring(0, 3).toUpperCase();

                html += `
                    <div class="team-badge" style="border-left: 3px solid ${color}; background:${bg}; color:${color}">
                        <span class="t-count">${count}</span>
                        <span class="t-name">${shortName}</span>
                    </div>
                `;
            });
            if (sortedTeams.length > 10) {
                const remaining = sortedTeams.length - 10;
                html += `<div class="team-badge" style="border-left: 3px solid #666; background:#333; color:#aaa; font-style:italic;">+${remaining}</div>`;
            }
        }
        html += `</div></div>`; 
    }

    document.getElementById('groups-container').innerHTML = html;
}

// --- MAIN ---
export async function main() {
    if (o101 && o101.initTeamColors) {
        try { await o101.initTeamColors(); } catch(e){}
    }

    loadSettings();

    document.getElementById('btn-settings').addEventListener('click', () => {
        const p = document.getElementById('settings-panel');
        p.style.display = p.style.display === 'block' ? 'none' : 'block';
    });
    document.getElementById('btn-save').addEventListener('click', saveSettings);
    
    document.getElementById('in-opacity').addEventListener('input', (e) => {
        settings.opacity = e.target.value; applyVisuals();
    });
    document.getElementById('in-zoom').addEventListener('input', (e) => {
        settings.zoom = e.target.value; applyVisuals();
    });
    
    document.getElementById('in-refresh').addEventListener('input', (e) => {
        settings.refreshRate = e.target.value;
        document.getElementById('refresh-label').innerText = settings.refreshRate + 's';
    });

    document.getElementById('in-nearby').addEventListener('input', (e) => {
        settings.nearbyCount = e.target.value;
        document.getElementById('nearby-label').innerText = settings.nearbyCount;
    });

    if (common && common.initInteractionListeners) {
        common.initInteractionListeners();
        common.subscribe('nearby', updateGroups);
    }
}