// Import du common système (toujours nécessaire pour la connexion)
import * as common from '/pages/src/common.mjs';

// Import du common "o101" (votre fichier local puissant)
// Assurez-vous que le fichier est bien dans src/o101/common.mjs
import * as o101 from './o101/common.mjs';

const listContainer = document.getElementById('list-container');
const headerRow = document.getElementById('header-row');
const dashEl = document.getElementById('dash');
const wrapperEl = document.getElementById('wrapper');
const checkboxesContainer = document.getElementById('cols-checkboxes');
const settingsPanel = document.getElementById('settings-panel');

// --- DEFINITION DES COLONNES ---
const ALL_COLS = [
    { id: 'flag',  label: 'NAT',  width: '40px', def: true,  align: 'center' },
    { id: 'cat',   label: 'CAT',  width: '40px', def: true,  align: 'center' },
    { id: 'grp',   label: 'GRP',  width: '35px', def: true,  align: 'center' },
    { id: 'name',  label: 'NOM',  width: '200px', def: true,  align: 'left' }, 
    { id: 'team',  label: 'CLUB', width: '95px', def: true,  align: 'left' },
    { id: 'draft', label: 'ASP',  width: '50px', def: false, align: 'right' },
    { id: 'watts', label: 'PWR',  width: '60px', def: true,  align: 'right' },
    { id: 'wkg',   label: 'WKG',  width: '55px', def: true,  align: 'right' },
    { id: 'bpm',   label: 'HR',   width: '45px', def: true,  align: 'right' },
    { id: 'gap',   label: 'GAP',  width: '65px', def: true,  align: 'right' }
];

let settings = {
    opacity: 90,
    zoom: 100,
    maxRiders: 30, 
    autoRiders: false, 
    showHeaders: true, 
    cols: {},      
    colWidths: {},
    colFontSizes: {},  
    panelPos: { x: 50, y: 50 } 
};

let calculatedMaxRiders = 30;

// --- GESTION REGLAGES ---
function loadSettings() {
    const sOp = localStorage.getItem('myMod_Coureur_Op');
    const sZoom = localStorage.getItem('myMod_Coureur_Zoom');
    
    settings.opacity = sOp ? parseInt(sOp) : 90;
    settings.zoom = sZoom ? parseInt(sZoom) : 100;

    const sMax = localStorage.getItem('myMod_Coureur_MaxRiders');
    settings.maxRiders = sMax ? parseInt(sMax) : 30;

    const sAuto = localStorage.getItem('myMod_Coureur_AutoRiders');
    settings.autoRiders = sAuto !== null ? (sAuto === 'true') : false;

    const sHeads = localStorage.getItem('myMod_Coureur_ShowHead');
    settings.showHeaders = sHeads !== null ? (sHeads === 'true') : true;

    const sCols = localStorage.getItem('myMod_Coureur_Cols_V6'); 
    if (sCols) settings.cols = JSON.parse(sCols);
    else ALL_COLS.forEach(c => settings.cols[c.id] = c.def);

    const sWidths = localStorage.getItem('myMod_Coureur_Widths');
    if (sWidths) settings.colWidths = JSON.parse(sWidths);

    const sFonts = localStorage.getItem('myMod_Coureur_ColFonts');
    if (sFonts) settings.colFontSizes = JSON.parse(sFonts);
    
    const sPos = localStorage.getItem('myMod_Coureur_PanelPos');
    if(sPos) settings.panelPos = JSON.parse(sPos);

    document.getElementById('in-opacity').value = settings.opacity;
    document.getElementById('in-zoom').value = settings.zoom;
    
    settingsPanel.style.left = settings.panelPos.x + 'px';
    settingsPanel.style.top = settings.panelPos.y + 'px';
    
    if (settings.autoRiders) calculateAutoRiders();

    renderSettingsUI();
    applyVisuals();
    updateGridLayout();
    initDraggablePanel();

    window.addEventListener('resize', () => {
        if (settings.autoRiders) {
            calculateAutoRiders();
        }
    });
}

function calculateAutoRiders() {
    const h = document.documentElement.clientHeight || window.innerHeight;
    const headerHeight = settings.showHeaders ? 40 : 5;
    const marginSafety = 20; 
    const availableHeight = h - headerHeight - marginSafety;
    const rowHeight = 38; 
    const count = Math.floor(availableHeight / rowHeight);
    calculatedMaxRiders = Math.min(Math.max(3, count), 50);
}

function saveSettings() {
    settings.opacity = parseInt(document.getElementById('in-opacity').value);
    settings.zoom = parseInt(document.getElementById('in-zoom').value);
    
    const inMax = document.getElementById('in-max-riders');
    if(inMax) settings.maxRiders = parseInt(inMax.value);

    const chkAuto = document.getElementById('chk-auto-riders');
    if(chkAuto) settings.autoRiders = chkAuto.checked;

    const chkHead = document.getElementById('chk-show-headers');
    if(chkHead) settings.showHeaders = chkHead.checked;

    ALL_COLS.forEach(c => {
        const chk = document.getElementById(`chk-${c.id}`);
        if(chk) settings.cols[c.id] = chk.checked;
    });
    
    localStorage.setItem('myMod_Coureur_Op', settings.opacity);
    localStorage.setItem('myMod_Coureur_Zoom', settings.zoom);
    localStorage.setItem('myMod_Coureur_MaxRiders', settings.maxRiders);
    localStorage.setItem('myMod_Coureur_AutoRiders', settings.autoRiders);
    localStorage.setItem('myMod_Coureur_ShowHead', settings.showHeaders);
    localStorage.setItem('myMod_Coureur_Cols_V6', JSON.stringify(settings.cols)); 
    localStorage.setItem('myMod_Coureur_Widths', JSON.stringify(settings.colWidths));
    localStorage.setItem('myMod_Coureur_ColFonts', JSON.stringify(settings.colFontSizes));
    localStorage.setItem('myMod_Coureur_PanelPos', JSON.stringify(settings.panelPos));

    if (settings.autoRiders) calculateAutoRiders();

    applyVisuals();
    updateGridLayout();
    settingsPanel.style.display = 'none';
}

function initDraggablePanel() {
    const handle = document.getElementById('panel-drag-handle');
    let isDragging = false;
    let startX, startY, initialLeft, initialTop;

    handle.addEventListener('mousedown', (e) => {
        isDragging = true;
        startX = e.clientX;
        startY = e.clientY;
        initialLeft = settingsPanel.offsetLeft;
        initialTop = settingsPanel.offsetTop;
        handle.style.cursor = 'grabbing';
    });

    document.addEventListener('mousemove', (e) => {
        if (!isDragging) return;
        const dx = e.clientX - startX;
        const dy = e.clientY - startY;
        settings.panelPos.x = initialLeft + dx;
        settings.panelPos.y = initialTop + dy;
        settingsPanel.style.left = settings.panelPos.x + 'px';
        settingsPanel.style.top = settings.panelPos.y + 'px';
    });

    document.addEventListener('mouseup', () => {
        if(isDragging) {
            isDragging = false;
            handle.style.cursor = 'grab';
        }
    });
}

function renderSettingsUI() {
    checkboxesContainer.innerHTML = '';
    
    const divHead = document.createElement('div');
    divHead.className = 'global-setting';
    divHead.innerHTML = `
        <div style="margin-bottom:5px;">
            <input type="checkbox" id="chk-show-headers" ${settings.showHeaders ? 'checked' : ''}>
            <label for="chk-show-headers" style="font-weight:bold; color:#fc6719;">AFFICHER LES ENTÊTES</label>
        </div>
    `;
    checkboxesContainer.appendChild(divHead);
    
    divHead.querySelector('#chk-show-headers').addEventListener('change', (e) => {
        settings.showHeaders = e.target.checked;
        updateGridLayout();
    });

    const divMax = document.createElement('div');
    divMax.className = 'input-group';
    const sliderDisplay = settings.autoRiders ? 'none' : 'block';
    
    divMax.innerHTML = `
        <div style="display:flex; justify-content:space-between; align-items:center;">
            <label>Nombre de coureurs : <span id="max-riders-label" style="color:white; font-weight:bold;">${settings.autoRiders ? 'AUTO' : settings.maxRiders}</span></label>
            <div style="display:flex; align-items:center;">
                <input type="checkbox" id="chk-auto-riders" ${settings.autoRiders ? 'checked' : ''}>
                <label for="chk-auto-riders" style="margin-left:5px; font-size:10px; cursor:pointer;">AUTO</label>
            </div>
        </div>
        <input type="range" id="in-max-riders" min="3" max="60" value="${settings.maxRiders}" style="display:${sliderDisplay}">
    `;
    checkboxesContainer.appendChild(divMax);

    const rangeInput = divMax.querySelector('#in-max-riders');
    const labelSpan = divMax.querySelector('#max-riders-label');
    const autoCheck = divMax.querySelector('#chk-auto-riders');

    rangeInput.addEventListener('input', (e) => {
        settings.maxRiders = parseInt(e.target.value);
        labelSpan.innerText = settings.maxRiders;
    });

    autoCheck.addEventListener('change', (e) => {
        settings.autoRiders = e.target.checked;
        if (settings.autoRiders) {
            rangeInput.style.display = 'none';
            labelSpan.innerText = 'AUTO';
            calculateAutoRiders();
        } else {
            rangeInput.style.display = 'block';
            labelSpan.innerText = settings.maxRiders;
        }
    });

    ALL_COLS.forEach(c => {
        const lbl = c.label || 'NAT'; 
        const checked = settings.cols[c.id] ? 'checked' : '';
        const isFlexible = c.width.includes('fr'); 
        
        let currentW = settings.colWidths[c.id];
        if (!currentW) currentW = parseInt(c.width) || 40; 
        let currentF = settings.colFontSizes[c.id] || 15; 

        let minW = 20, maxW = 150;
        if (c.id === 'name') { minW = 50; maxW = 400; }
        if (c.id === 'team') { minW = 30; maxW = 250; }
        if (c.id === 'grp')  { minW = 20; maxW = 80; } 

        let widthSliderHtml = isFlexible 
            ? `<span class="locked-val">AUTO</span>`
            : `<input type="range" class="mini-slider" id="sli-w-${c.id}" min="${minW}" max="${maxW}" value="${currentW}" title="Largeur: ${currentW}px">`;

        const div = document.createElement('div');
        div.className = 'col-config-block';
        div.innerHTML = `
            <div class="col-head">
                <input type="checkbox" id="chk-${c.id}" ${checked}>
                <label for="chk-${c.id}">${lbl}</label>
            </div>
            <div class="col-params">
                <div class="param-row">
                    <span class="p-lbl">Larg.</span>
                    ${widthSliderHtml}
                </div>
                <div class="param-row">
                    <span class="p-lbl">Txt</span>
                    <input type="range" class="mini-slider" id="sli-f-${c.id}" min="10" max="25" value="${currentF}" title="Taille: ${currentF}px">
                </div>
            </div>
        `;
        checkboxesContainer.appendChild(div);

        if (!isFlexible) {
            div.querySelector(`#sli-w-${c.id}`).addEventListener('input', (e) => {
                settings.colWidths[c.id] = parseInt(e.target.value);
                updateGridLayout(); 
            });
        }
        div.querySelector(`#sli-f-${c.id}`).addEventListener('input', (e) => {
            settings.colFontSizes[c.id] = parseInt(e.target.value);
        });
        div.querySelector(`#chk-${c.id}`).addEventListener('change', (e) => {
             settings.cols[c.id] = e.target.checked;
             updateGridLayout();
        });
    });
}

function updateGridLayout() {
    headerRow.style.display = settings.showHeaders ? 'grid' : 'none';
    const activeCols = ALL_COLS.filter(c => settings.cols[c.id]);
    let gridTemplateParts = [];
    let totalFixedWidth = 0;

    activeCols.forEach(c => {
        if (c.width.includes('fr')) {
            gridTemplateParts.push('1fr');
            totalFixedWidth += 200; 
        } else {
            const w = settings.colWidths[c.id] || parseInt(c.width);
            gridTemplateParts.push(`${w}px`);
            totalFixedWidth += w;
        }
    });

    const totalW = totalFixedWidth + 10; 
    dashEl.style.width = `${Math.max(300, totalW)}px`;
    const gridTemplate = gridTemplateParts.join(' ');
    dashEl.style.setProperty('--grid-layout', gridTemplate);

    let headerHtml = '';
    activeCols.forEach(c => {
        headerHtml += `<div class="h-col ${c.align}">${c.label}</div>`;
    });
    headerRow.innerHTML = headerHtml;
}

function applyVisuals() {
    document.getElementById('op-label').innerText = settings.opacity;
    dashEl.style.backgroundColor = `rgba(0, 0, 0, ${settings.opacity/100})`;
    listContainer.style.backgroundColor = `rgba(0, 0, 0, ${settings.opacity/100})`;
    document.getElementById('zoom-label').innerText = (settings.zoom/100).toFixed(1);
    wrapperEl.style.transform = `scale(${settings.zoom/100})`;
}

// --- UTILITAIRES ---
function formatTime(sec) {
    if (sec == null || isNaN(sec)) return '';
    const abs = Math.abs(sec);
    if (abs < 60) return Math.round(abs) + 's';
    const m = Math.floor(abs / 60);
    const s = Math.floor(abs % 60);
    return `${m}:${s.toString().padStart(2,'0')}`;
}

function getCountryCodeText(countryCode) {
    if (!countryCode) return '';
    return String(countryCode).toUpperCase();
}

function formatName(first, last) {
    if (o101.fmtFirstName && o101.fmtLastName) {
        return o101.fmtFirstName({athlete:{firstName:first}}, 1) + '. ' + o101.fmtLastName({athlete:{lastName:last}});
    }
    if (!last) return first || 'Inconnu';
    let cleanLast = last.replace(/\[.*?\]|\(.*?\)/g, '').trim();
    const fInitial = first ? first.charAt(0).toUpperCase() + '.' : '';
    return `${fInitial} ${cleanLast.toUpperCase()}`;
}

function getTeamColor(teamName) {
    if (!teamName) return '#FFFFFF';
    if (o101.preferredTeamColor) {
        const tc = o101.preferredTeamColor(teamName);
        if (tc && tc.color) {
            if (tc.color !== '#FFF' && tc.color !== '#FFFFFF') {
                return tc.color;
            }
        }
    }
    return stringToColorHSL(teamName);
}

function stringToColorHSL(str) {
    let hash = 0;
    for (let i = 0; i < str.length; i++) {
        hash = str.charCodeAt(i) + ((hash << 5) - hash);
    }
    const h = Math.abs(hash) % 360;
    const s = 80;
    const l = 70;
    return `hsl(${h}, ${s}%, ${l}%)`;
}

// --- LOGIQUE ZWIFT ---
function updateRiders(data) {
    let riders = [];
    if (Array.isArray(data)) riders = data;
    else if (data && data.riders) riders = data.riders;
    else if (data && data.nearby) riders = data.nearby;
    else return;

    if (riders.length === 0) {
        listContainer.innerHTML = '<div class="status-msg">Aucun coureur proche</div>';
        return;
    }

    if (settings.autoRiders) {
        calculateAutoRiders();
    }

    let myTeamName = null;
    const me = riders.find(r => r.watching);
    if (me && me.athlete && me.athlete.team) {
        myTeamName = me.athlete.team.trim().toUpperCase();
    }

    riders.sort((a, b) => Math.abs(a.gap || 0) - Math.abs(b.gap || 0));
    
    const limit = settings.autoRiders ? calculatedMaxRiders : settings.maxRiders;
    let visible = riders.slice(0, limit);
    
    visible.sort((a, b) => (a.gap || 0) - (b.gap || 0));

    let groups = [];
    if (visible.length > 0) {
        let currentGroup = [visible[0]];
        for (let i = 1; i < visible.length; i++) {
            const r = visible[i];
            const prev = visible[i-1];
            const diff = (r.gap || 0) - (prev.gap || 0);

            if (diff > 4) {
                groups.push(currentGroup);
                currentGroup = [r];
            } else {
                currentGroup.push(r);
            }
        }
        groups.push(currentGroup);
    }

    let html = '';
    const activeCols = ALL_COLS.filter(c => settings.cols[c.id]);
    let groupAlt = false; 

    groups.forEach((group, groupIndex) => {
        
        // --- CALCUL WKG MOYEN DU GROUPE ---
        let avgWkg = 0;
        let groupTeams = new Map();

        if (group.length > 0) {
            let totalWkg = 0;
            group.forEach(r => {
                const p = (r.state && r.state.power) || 0;
                const w = (r.athlete && r.athlete.weight) || 75;
                totalWkg += (p/w);

                if (r.athlete && r.athlete.team) {
                    const tName = r.athlete.team.trim();
                    if (tName) {
                        groupTeams.set(tName, (groupTeams.get(tName) || 0) + 1);
                    }
                }
            });
            avgWkg = (totalWkg / group.length).toFixed(1);
        }

        // --- PREPARATION AFFICHAGE EQUIPES DANS LE SEPARATEUR ---
        let teamBadgesHtml = '';
        const sortedGroupTeams = [...groupTeams.entries()].sort((a, b) => b[1] - a[1]);
        
        // Afficher les top 10 équipes du groupe (au lieu de 3)
        sortedGroupTeams.slice(0, 10).forEach(([name, count]) => {
            const color = getTeamColor(name);
            // Badge simple avec juste le chiffre encadré de la couleur
            teamBadgesHtml += `<span style="border:1px solid ${color}; color:${color}; background:${color}22; padding:0 3px; border-radius:3px; margin-left:4px; font-size:10px; font-weight:bold;" title="${name}">${count}</span>`;
        });


        // --- AFFICHAGE SEPARATEUR ---
        let labelGap = '';
        let classGap = 'gap-count';

        if (groupIndex === 0) {
            labelGap = '<span style="color:#44ff44; font-weight:900;">TÊTE</span>';
        } else {
            const prevGroup = groups[groupIndex - 1];
            const lastOfPrev = prevGroup[prevGroup.length - 1];
            const firstOfCurr = group[0];
            const diff = (firstOfCurr.gap || 0) - (lastOfPrev.gap || 0);
            
            groupAlt = !groupAlt;
            
            labelGap = `<span class="gap-time">+ ${formatTime(diff)}</span>`;
        }

        html += `<div class="gap-separator">
                    ${labelGap}
                    <span class="${classGap}">${group.length} 👤</span>
                    <span class="gap-wkg" style="background:#333; color:#aaa; font-family:monospace; padding:1px 6px; border-radius:10px; font-size:11px;">${avgWkg} w/kg</span>
                    ${teamBadgesHtml}
                 </div>`;

        // --- AFFICHAGE COUREURS ---
        group.forEach(r => {
            let d = {}; 
            d.flag = ''; d.team = '-'; d.cat = '';
            d.grp = group.length;

            let isMyTeammate = false;
            let teamColor = '#FFFFFF'; 

            if (r.athlete) {
                d.name = formatName(r.athlete.firstName, r.athlete.lastName);
                
                let flagContent = null;
                if (o101.fmtFlag) {
                    try {
                        flagContent = o101.fmtFlag({ athlete: r.athlete });
                    } catch(e) {}
                }
                
                if (flagContent) {
                    d.flag = flagContent;
                } else {
                    let cc = r.athlete.countryAlpha2 || r.athlete.countryCode;
                    d.flag = getCountryCodeText(cc);
                }

                if (r.athlete.team) {
                    d.team = r.athlete.team;
                    teamColor = getTeamColor(d.team);
                    
                    if (myTeamName && r.athlete.team.trim().toUpperCase() === myTeamName && !r.watching) {
                        isMyTeammate = true;
                    }
                }
            } else { d.name = 'Inconnu'; }
    
            if (r.category) d.cat = r.category;
            else if (r.subgroup) d.cat = r.subgroup; 
            else if (r.athlete && r.athlete.category) d.cat = r.athlete.category;
            
            const pwr = (r.state && r.state.power) || 0;
            const wgt = (r.athlete && r.athlete.weight) || 75;
            const wkg = (pwr / wgt);
            
            d.watts = pwr > 0 ? Math.round(pwr) : '';
            d.wkg = wkg > 0 ? wkg.toFixed(1) : '';
            d.bpm = (r.state && r.state.heartrate) > 0 ? r.state.heartrate : '';
    
            const draft = (r.state && r.state.draft) || 0;
            d.draft = draft > 5 ? Math.round(draft) : ''; 
    
            const gap = r.gap || 0;
            if (gap === 0) {
                d.gap = '';
            } else {
                const timeStr = formatTime(gap);
                const sign = gap > 0 ? '+' : ''; 
                const gapColor = gap > 0 ? '#ff5555' : '#55ff55';
                d.gap = `${sign}<span style="color:${gapColor}">${timeStr}</span>`;
            }
    
            let z = 'z1';
            if (wkg >= 6.0) z='z6'; else if (wkg >= 4.5) z='z5'; else if (wkg >= 3.7) z='z4';
            else if (wkg >= 3.2) z='z3'; else if (wkg >= 2.5) z='z2';
    
            let rowClass = 'rider-row';
            if (r.watching) rowClass += ' watching'; 
            else if (isMyTeammate) rowClass += ' teammate';
            else if (groupAlt) rowClass += ' group-alt'; 
            
            html += `<div class="${rowClass}">`;
            
            activeCols.forEach(col => {
                let content = d[col.id];
                let classes = `cell ${col.align}`;
                let titleText = String(content).replace(/<[^>]*>?/gm, '');
                
                const fontSize = settings.colFontSizes[col.id] || 15;
                let style = `font-size:${fontSize}px;`;
                
                if (col.id === 'flag') classes += ' r-flag'; 
                if (col.id === 'cat') classes += ' r-cat';   
                if (col.id === 'grp') classes += ' r-grp'; 
                if (col.id === 'name') classes += ' r-name';
                
                if (col.id === 'team') {
                    classes += ' r-team';
                    if (content !== '-') {
                        // STYLE ENCADRÉ (Bordure 1px + Fond léger + Texte coloré)
                        style += `
                            border: 1px solid ${teamColor}; 
                            color: ${teamColor};
                            background: ${teamColor}15;
                            border-radius: 4px;
                            padding: 0 4px;
                            font-weight: 600;
                        `;
                    }
                }

                if (col.id === 'draft') classes += ' r-draft'; 
                if (col.id === 'watts') classes += ' r-watts';
                if (col.id === 'bpm') classes += ' r-bpm';
                if (col.id === 'wkg') classes += ' r-wkg ' + z;
                if (col.id === 'gap') classes += ' r-gap';
    
                html += `<div class="${classes}" style="${style}" title="${titleText}">${content}</div>`;
            });
            html += `</div>`;
        });
    });

    listContainer.innerHTML = html;
}

export async function main() {
    if (o101.initNationFlags) {
        try {
            await o101.initNationFlags();
        } catch (e) {
            console.warn("Erreur Init O101 Flags:", e);
        }
    }
    if (o101.initTeamColors) {
        try {
            await o101.initTeamColors();
        } catch (e) {
            console.warn("Erreur Init O101 TeamColors:", e);
        }
    }

    loadSettings();
    document.getElementById('btn-settings').addEventListener('click', () => {
        const p = document.getElementById('settings-panel');
        p.style.display = p.style.display === 'block' ? 'none' : 'block';
    });
    document.getElementById('btn-save').addEventListener('click', saveSettings);
    
    document.getElementById('in-opacity').addEventListener('input', (e) => {
        settings.opacity = e.target.value; applyVisuals();
    });
    document.getElementById('in-zoom').addEventListener('input', (e) => {
        settings.zoom = e.target.value; applyVisuals();
    });

    if (common && common.initInteractionListeners) {
        common.initInteractionListeners();
        common.subscribe('nearby', updateRiders);
    }
}